// SPDX-License-Identifier: BSD-3-Clause
//
// Copyright 2010 Raritan Inc. All rights reserved.

package com.raritan.util;

import java.util.ArrayList;
import java.util.List;

/**
 * Tracks an asynchronous activity.
 *
 * <p>
 * Asynchronous object methods usually return this as a handle for the task that
 * they are going to perform. They also keep the handle themselves, and once
 * they're done, they call succeeded() or failed() on it, to make it go to
 * success/failure state.
 * </p>
 *
 * <p>
 * Users of this can call addSuccessListener() or addFailureListener() at any
 * time, to be notified when this goes to or already is in success/failure
 * state. Notification will always happen asynchronously, i.e. via event loop,
 * and not via a direct recursive call from the add...() methods above.
 * </p>
 *
 * Note that this is only <em>tracks</em> an activity, but does provide the
 * implementation itself. Compare to: {@link AsyncCommand}.
 */
public class AsyncRequest {
    private static final String CLASS_NAME = "AsyncRequest";

    private enum State {
        NEW,       // task just created - not running, yet
        STARTED,   // task is running
        SUCCESS,   // task succeeded - final state
        FAILURE,   // task failed - final state
    };

    private State m_state = State.NEW;
    private Object m_successData;
    private Exception m_failureException;
    private List<AsyncSuccessListener> m_succListeners = new ArrayList<AsyncSuccessListener>();
    private List<AsyncFailureListener> m_failListeners = new ArrayList<AsyncFailureListener>();
    private List<AsyncDoneListener> m_doneListeners = new ArrayList<AsyncDoneListener>();
    private String m_label;

    /**
     * Constructor.
     *
     * @param label  a (unique) string to associate with this instance - for debugging
     */
    public AsyncRequest(String label) {
        m_label = label;
    }

    /**
     * Add a listener to notify in case this task succeeds.
     *
     * If it already has succeeded, listener will still be notified.
     * Notification is guaranteed to happen asynchronously.
     */
    public void addSuccessListener(AsyncSuccessListener l) {
        if (hasSucceeded()) {
            l.onSuccess(m_successData);
        } else {
            m_succListeners.add(l);
        }
    }

    /**
     * Add a listener to notify in case this task fails.
     *
     * If it already has failed, listener will still be notified.
     * Notification is guaranteed to happen asynchronously.
     */
    public void addFailureListener(final AsyncFailureListener l) {
        if (hasFailed()) {
            l.onFailure(m_failureException);
        } else {
            m_failListeners.add(l);
        }
    }

    /**
     * Add a listener to notify when task has either succeeded or failed.
     *
     * If it already has failed, listener will still be notified.
     * Notification is guaranteed to happen asynchronously.
     */
    public void addDoneListener(final AsyncDoneListener l) {
        if (isDone()) {
            l.onDone();
        } else {
            m_doneListeners.add(l);
        }
    }

    /**
     * Inform this instance that the task it represents has started to run.
     *
     * @return this instance (for syntactic convenience)
     */
    public AsyncRequest started() {
        assert m_state == State.NEW;
        m_state = State.STARTED;
        return this;
    }

    /**
     * Inform this instance that the task it represents has succeeded.
     *
     * @param data  context data to pass to registered success listeners
     * @return this instance (for syntactic convenience)
     */
    public AsyncRequest succeeded(Object data) {
        assert isNew() || hasStarted();
        m_state = State.SUCCESS;
        m_successData = data;

        for (AsyncSuccessListener sl : m_succListeners) {
            sl.onSuccess(data);
        }
        for (AsyncDoneListener dl : m_doneListeners) {
            dl.onDone();
        }
        return this;
    }

    /**
     * Inform this instance that the task it represents has failed.
     *
     * @param exc  exception to pass to registered failure listeners
     */
    public AsyncRequest failed(Exception exc) {
        assert isNew() || hasStarted();
        m_state = State.FAILURE;
        m_failureException = exc;

        for (AsyncFailureListener fl : m_failListeners) {
            fl.onFailure(exc);
        }
        for (AsyncDoneListener dl : m_doneListeners) {
            dl.onDone();
        }
        return this;
    }

    /**
     * Returns an adapter listener, that can be registered with
     * addSuccessListener() to some other AsyncRequest. Then, once the other task
     * goes to success state, this task will automatically go to success state,
     * too.
     */
    public AsyncSuccessListener getMakeThisSucceedAdapter() {
        return new AsyncSuccessListener() {
            @Override
            public void onSuccess(Object data) {
                AsyncRequest.this.succeeded(data);
            }
        };
    }

    /**
     * Returns an adapter listener, that can be registered with
     * addFailureListener() to some other AsyncRequest. Then, once the other task
     * goes to failure state, this task will automatically go to failure state,
     * too.
     */
    public AsyncFailureListener getMakeThisFailAdapter() {
        return new AsyncFailureListener() {
            @Override
            public void onFailure(Exception exc) {
                AsyncRequest.this.failed(exc);
            }
        };
    }

    public boolean isNew() {
        return m_state == State.NEW;
    }

    public boolean hasStarted() {
        return m_state == State.STARTED;
    }

    public boolean hasFailed() {
        return m_state == State.FAILURE;
    }

    public boolean hasSucceeded() {
        return m_state == State.SUCCESS;
    }

    public boolean isDone() {
        return hasSucceeded() || hasFailed();
    }

    public String getLabel() {
        return m_label;
    }

    public Object getSuccessData() {
        return m_successData;
    }

    public Exception getFailureException() {
        return m_failureException;
    }

    @Override
    public String toString() {
        return CLASS_NAME + "@" + Integer.toHexString(hashCode()) +
                "(label = '" + m_label + "', state = " + m_state + ")";
    }
}

