// SPDX-License-Identifier: BSD-3-Clause
//
// Copyright 2010 Raritan Inc. All rights reserved.

package com.raritan.util;

import java.util.ArrayList;
import java.util.List;

/**
 * Tracks multiple {@link AsyncTask}'s tasks running in parallel.
 *
 * It will go to success state, when all tasks tracked by this instance have
 * succeeded. It will immediately go to failure state, when any of the tasks has
 * failed.
 *
 * Note that this is only _tracks_ activities, but does not manage or provide
 * the implementations of them, as opposed to {@link AsyncCommand}.
 */
public class AsyncMultiRequest extends AsyncRequest {
    private List<AsyncRequest> m_requests = new ArrayList<AsyncRequest>();
    private List<AsyncRequest> m_failed = new ArrayList<AsyncRequest>();
    private List<AsyncRequest> m_succeeded = new ArrayList<AsyncRequest>();
    int tasksPending = 0;

    /**
     * Constructor.
     *
     * @param label  a (unique) string to associate with this task - for debugging
     */
    public AsyncMultiRequest(String label) {
        super(label);
    }

    /**
     * Add an {@link AsyncTask} to the list of tracked tasks.
     */
    public void add(final AsyncRequest request) {
        assert isNew();

        m_requests.add(request);
        tasksPending++;

        request.addSuccessListener(new AsyncSuccessListener() {
            @Override
            public void onSuccess(Object data) {
                m_succeeded.add(request);
                if (--tasksPending == 0 && hasStarted()) allDone();
            }
        });
        request.addFailureListener(new AsyncFailureListener() {
            @Override
            public void onFailure(Exception exc) {
                m_failed.add(request);
                if (--tasksPending == 0 && hasStarted()) allDone();
            }
        });
    }

    /**
     * Inform this instance that all tasks have been added and tracking can start.
     *
     * Note about running success/failure commands (SFCs):
     * We cannot call SFCs before this method, because we don't know about the
     * state of any tasks added in the future.  However, once this method is
     * called, we imply (and enforce) that no further tasks will be added.
     * Hence, SFCs can then be called once all tasks have finished.
     *
     * The above handles the following corner-cases gracefully:
     * 1. Multiple already-finished tasks can be added w/o calling SFCs too early.
     * 2. It's possible to add no tasks at all and still have SFCs called.
     */
    @Override
    public AsyncRequest started() {
        super.started();
        if (tasksPending == 0) allDone();
        return this;
    }

    public AsyncRequest succeeded(Object data) {
        assert false;  // not for public use
        return this;
    }
    public AsyncRequest failed(Exception exc) {
        assert false;  // not for public use
        return this;
    }

    private void allDone() {
        if (m_failed.isEmpty()) {
            // this includes the "no tasks added" case
            super.succeeded(m_succeeded);
        } else {
            super.failed(new AsyncMultiRequestFailure(m_failed));
        }
    }

    public List<AsyncRequest> getFailedRequests() {
        return m_failed;
    }
}
