// SPDX-License-Identifier: BSD-3-Clause
//
// Copyright 2012 Raritan Inc. All rights reserved.

package com.raritan.json_rpc;

import org.json.JSONException;
import org.json.JSONObject;

import com.raritan.idl.AsyncRpcResponse;
import com.raritan.idl.RpcCtrl;
import com.raritan.idl.TypeInfo;
import com.raritan.util.AsyncRequest;
import com.raritan.util.StringUtils;

public class ObjectProxy {

    protected final Agent agent;
    protected final String rid;
    protected TypeInfo typeInfo;

    private static final String JSONRPC_KEY = "jsonrpc";
    private static final String JSONRPC_VERSION = "2.0";
    private static final String METHOD_KEY = "method";
    private static final String PARAMS_KEY = "params";
    private static final String ID_KEY = "id";
    private static final String RESULT_KEY = "result";
    private static final String ERROR_KEY = "error";
    private static final String ERROR_CODE_KEY = "code";
    private static final String ERROR_MESSAGE_KEY = "message";

    public ObjectProxy(Agent agent, String rid, TypeInfo typeInfo) {
        this.agent = agent;
        this.rid = rid;
        this.typeInfo = typeInfo;
    }

    public ObjectProxy(Agent agent, String rid) {
        this(agent, rid, null);
    }

    public ObjectProxy(ObjectProxy proxy) {
        this.agent = proxy.agent;
        this.rid = proxy.rid;
        this.typeInfo = proxy.typeInfo;
    }

    @Override
    public int hashCode() {
        return rid.hashCode();
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof ObjectProxy) {
            ObjectProxy otherProxy = (ObjectProxy) other;
            if (!StringUtils.stringsEqual(rid, otherProxy.rid)) {
                return false;
            }
            if (!agent.equals(otherProxy.agent)) {
                return false;
            }
            return true;
        }

        return false;
    }

    @Override
    public String toString() {
        return rid;
    }

    public Agent getAgent() {
        return agent;
    }

    public String getRid() {
        return rid;
    }

    private static class JsonRpcError {
        public int code;
        public String message;

        public JsonRpcError(JSONObject json) throws JSONException {
            code = json.getInt(ERROR_CODE_KEY);
            message = json.getString(ERROR_MESSAGE_KEY);
        }
    };

    private static class MethodResults {
        public String jsonRpcVersion = null;
        public int id = -1;
        public JSONObject result = null;
        public JsonRpcError error = null;

        public MethodResults(JSONObject json) {
            try {
                jsonRpcVersion = json.getString(JSONRPC_KEY);
                id = json.getInt(ID_KEY);
                if (json.has(ERROR_KEY)) {
                    error = new JsonRpcError(json.getJSONObject(ERROR_KEY));
                } else if (json.has(RESULT_KEY)) {
                    result = json.getJSONObject(RESULT_KEY);
                }
            } catch (JSONException e) {
            }
        }

        public boolean isValid(int expectedId) {
            return JSONRPC_VERSION.equals(jsonRpcVersion) && id == expectedId;
        }
    }

    protected JSONObject rpcCall(String methodName, JSONObject params) throws Exception {
        int id = agent.getNextRequestId();

        JSONObject reqJson = new JSONObject();
        reqJson.put(JSONRPC_KEY, JSONRPC_VERSION);
        reqJson.put(METHOD_KEY, methodName);
        reqJson.put(PARAMS_KEY, params);
        reqJson.put(ID_KEY, id);

        String response = agent.performRequest(rid, reqJson.toString());
        JSONObject rspJson = new JSONObject(response);

        MethodResults results = new MethodResults(rspJson);
        if (!results.isValid(id)) {
            throw new RpcFormatException();
        } else if (results.error != null) {
            throw new RpcErrorException(results.error.code, results.error.message);
        }
        return results.result;
    }

    protected AsyncRequest rpcCall(final String methodName, final JSONObject params, final AsyncRpcResponse<JSONObject> rsp, RpcCtrl rpcCtrl) {
        if (rpcCtrl.getBulk() == RpcCtrl.Bulk.ALLOW) {
            final int id = agent.getNextRequestId();
            JSONObject reqJson = new JSONObject();

            try {
                reqJson.put(JSONRPC_KEY, JSONRPC_VERSION);
                reqJson.put(METHOD_KEY, methodName);
                reqJson.put(PARAMS_KEY, params);
                reqJson.put(ID_KEY, id);
            } catch (JSONException e) {
                rsp.onFailure(e);
            }

            BulkRequestQueue queue = agent.getDefaultBulkRequestQueue();
            return queue.enqueueRequest(rid, reqJson, new BulkRequestQueue.JSONDecoder() {
                @Override
                public JSONObject decodeAndValidate(String json) throws JSONException, RpcException {
                    JSONObject rspJson = new JSONObject(json);
                    MethodResults results = new MethodResults(rspJson);

                    if (!results.isValid(id)) {
                        throw new RpcFormatException();
                    } else if (results.error != null) {
                        throw new RpcErrorException(results.error.code, results.error.message);
                    }

                    return results.result;
                }
            }, rsp);
        } else {
            final String label = typeInfo + "::" + methodName;
            final AsyncRequest request = new AsyncRequest(label);

            agent.scheduleInBackground(new Runnable() {
                @Override
                public void run() {
                    try {
                        JSONObject json = rpcCall(methodName, params);
                        rsp.onSuccess(json);
                        request.succeeded(json);
                    } catch (Exception e) {
                        rsp.onFailure(e);
                        request.failed(e);
                    }
                }
            });
            return request;
        }
    }

    private TypeInfo decodeTypeInfo(JSONObject json) throws JSONException {
        String id = json.getString("id");
        TypeInfo base = null;
        if (!json.isNull("base")) {
            base = decodeTypeInfo(json.getJSONObject("base"));
        }
        return new TypeInfo(id, base);
    }

    public TypeInfo getTypeInfo() throws Exception {
        if (typeInfo == null) {
            JSONObject result = rpcCall("getTypeInfo", null);
            typeInfo = decodeTypeInfo(result.getJSONObject("_ret_"));
        }
        return typeInfo;
    }

    public AsyncRequest getTypeInfo(AsyncRpcResponse<TypeInfo> rsp) {
        return getTypeInfo(rsp, RpcCtrl.getDefault());
    }

    public AsyncRequest getTypeInfo(final AsyncRpcResponse<TypeInfo> rsp, final RpcCtrl rpcCtrl) {
        return rpcCall("getTypeInfo", null, new AsyncRpcResponse<JSONObject>() {
            @Override
            public void onSuccess(JSONObject result) {
                try {
                    typeInfo = decodeTypeInfo(result.getJSONObject("_ret_"));
                    rsp.onSuccess(typeInfo);
                } catch (Exception e) {
                    onFailure(e);
                }
            }

            @Override
            public void onFailure(Exception e) {
                rsp.onFailure(e);
            }
        }, rpcCtrl);
    }

    public TypeInfo getStaticTypeInfo() {
        if (typeInfo != null) {
            return typeInfo;
        } else {
            return new TypeInfo("idl.Object:1.0.0", null);
        }
    }

    private static TypeInfo findCompatibleTypeInfo(String typeId) {
        TypeInfo requestedType = new TypeInfo(typeId, null);
        TypeInfo bestTypeInfo = null;

        for (TypeInfo ti : TypeInfoLib.typeInfos) {
            if (requestedType.isCompatible(ti)) {
                if (bestTypeInfo == null || ti.compareTo(bestTypeInfo) > 0) {
                    bestTypeInfo = ti;
                }
            }
        }

        return bestTypeInfo;
    }

    public static ObjectProxy decode(JSONObject json, Agent agent) throws JSONException {
        if (json == null) {
            return null;
        }

        // typeInfo may be null if no compatible type was found, so we can't determine
        // the hierarchy. It will be queried from the server in the next call to
        // getTypeInfo().
        TypeInfo typeInfo = findCompatibleTypeInfo(json.getString("type"));
        return new ObjectProxy(agent, json.getString("rid"), typeInfo);
    }

    public Object encode() throws JSONException {
        JSONObject json = new JSONObject();
        json.put("type", getStaticTypeInfo());
        json.put("rid", getRid());
        return json;
    }

}
