// SPDX-License-Identifier: BSD-3-Clause
//
// Copyright 2012 Raritan Inc. All rights reserved.

package com.raritan.idl;

import com.raritan.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;

public class TypeInfo implements Comparable<TypeInfo> {

    private final String typename;
    private final int major;
    private final int submajor;
    private final int minor;
    private final TypeInfo base;
    private ArrayList<TypeInfo> segments;

    public TypeInfo(String typename, int major, int submajor, int minor) {
        this.typename = typename;
        this.major = major;
        this.submajor = submajor;
        this.minor = minor;
        this.base = null;
    }

    public TypeInfo(String typeName, TypeInfo baseType) {
        String[] tokens = typeName.split(":");
        typename = tokens[0];
        tokens = tokens[1].split("\\.");
        major = Integer.valueOf(tokens[0]);
        submajor = Integer.valueOf(tokens[1]);
        minor = Integer.valueOf(tokens[2]);
        base = baseType;
    }

    public String getTypeName() {
        return typename;
    }

    public TypeInfo getBase() {
        return base;
    }

    @Override
    public String toString() {
        return typename + ":" + major + "." + submajor + "." + minor;
    }

    @Override
    public int hashCode() {
        int hash = 3;

        hash += 5 * typename.hashCode();
        hash += 7 * major;
        hash += 9 * submajor;
        hash += 11 * minor;

        return hash;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof TypeInfo) {
            TypeInfo otherInfo = (TypeInfo) other;
            return StringUtils.stringsEqual(typename, otherInfo.typename) &&
                   major == otherInfo.major &&
                   submajor == otherInfo.submajor &&
                   minor == otherInfo.minor;
        }
        return false;
    }

    @Override
    public int compareTo(TypeInfo other) {
        int result = typename.compareTo(other.typename);
        if (result != 0) {
            return result;
        }
        if (major != other.major) {
            return major < other.major ? -1 : 1;
        }
        if (submajor != other.submajor) {
            return submajor < other.submajor ? -1 : 1;
        }
        if (minor != other.minor) {
            return minor < other.minor ? -1 : 1;
        }
        return 0;
    }

    // returns the type name with all version numbers removed
    public String getStrippedTypeName() {
        ensureSegments();
        StringBuilder s = new StringBuilder();
        for (TypeInfo segment : segments) {
            if (s.length() > 0) s.append('.');
            s.append(segment.typename);
        }
        return s.toString();
    }

    public boolean isCompatible(TypeInfo other) {
        ensureSegments();
        other.ensureSegments();
        if (segments.size() != other.segments.size()) {
            return false;
        }
        for (int i = 0; i < segments.size(); i++) {
            TypeInfo s = segments.get(i);
            TypeInfo os = other.segments.get(i);
            if (!StringUtils.stringsEqual(s.typename, os.typename) ||
                    s.major != os.major ||
                    s.submajor != os.submajor ||
                    s.minor < os.minor) {
                return base != null && base.isCompatible(other);
            }
        }

        return true;
    }

    public boolean isCallCompatible(TypeInfo other) {
        ensureSegments();
        other.ensureSegments();
        if (segments.size() != other.segments.size()) {
            return false;
        }
        for (int i = 0; i < segments.size(); i++) {
            TypeInfo s = segments.get(i);
            TypeInfo os = other.segments.get(i);
            if (!StringUtils.stringsEqual(s.typename, os.typename) ||
                    s.major != os.major ||
                    s.minor < os.minor) {
                return base != null && base.isCallCompatible(other);
            }
        }

        return true;
    }

    private void ensureSegments() {
        if (segments != null) {
            return;
        }
        segments = new ArrayList<TypeInfo>();

        String[] parts = typename.split("\\.");
        for (int i = 0; i < parts.length - 1; i++) {
            String[] nameElems = parts[i].split("_");
            int maj, submaj, min;

            if (nameElems.length >= 4) {
                try {
                    maj = Integer.valueOf(nameElems[nameElems.length - 3]);
                    submaj = Integer.valueOf(nameElems[nameElems.length - 2]);
                    min = Integer.valueOf(nameElems[nameElems.length - 1]);

                    nameElems = Arrays.copyOf(nameElems, nameElems.length - 3);
                    String actualName = StringUtils.join(nameElems, "_");
                    segments.add(new TypeInfo(actualName, maj, submaj, min));
                    continue;
                } catch (NumberFormatException e) {
                }
            }
            segments.add(new TypeInfo(parts[i], 1, 0, 0));
        }

        segments.add(new TypeInfo(parts[parts.length - 1], major, submajor, minor));
    }
}
